import React from 'react';
import { useQuery } from '@tanstack/react-query';
import { api } from '../services/api';
import {
  TrendingUp,
  ShoppingCart,
  Users,
  DollarSign,
  Package,
  Calendar,
  BarChart3,
  Activity
} from 'lucide-react';
import {
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  BarChart,
  Bar,
  PieChart,
  Pie,
  Cell
} from 'recharts';

export default function Dashboard() {
  const { data: stats } = useQuery({
    queryKey: ['dashboard-stats'],
    queryFn: async () => {
      const response = await api.get('/orders/reports/daily');
      return response.data.data;
    }
  });

  const { data: recentOrders } = useQuery({
    queryKey: ['recent-orders'],
    queryFn: async () => {
      const response = await api.get('/orders?limit=5');
      return response.data.data;
    }
  });

  const { data: topProducts } = useQuery({
    queryKey: ['top-products'],
    queryFn: async () => {
      const response = await api.get('/products?limit=5');
      return response.data.data;
    }
  });

  // Mock data for charts
  const salesData = [
    { name: 'Pzt', sales: 1200 },
    { name: 'Sal', sales: 1900 },
    { name: 'Çar', sales: 1500 },
    { name: 'Per', sales: 2100 },
    { name: 'Cum', sales: 2800 },
    { name: 'Cmt', sales: 3200 },
    { name: 'Paz', sales: 1800 },
  ];

  const categoryData = [
    { name: 'Ana Yemekler', value: 35, color: '#ef4444' },
    { name: 'İçecekler', value: 25, color: '#dc2626' },
    { name: 'Tatlılar', value: 20, color: '#b91c1c' },
    { name: 'Salatalar', value: 15, color: '#991b1b' },
    { name: 'Diğer', value: 5, color: '#7f1d1d' },
  ];

  const statCards = [
    {
      title: 'Günlük Ciro',
      value: stats?.totalSales ? `₺${stats.totalSales.toLocaleString()}` : '₺0',
      change: '+12.5%',
      icon: DollarSign,
      color: 'primary'
    },
    {
      title: 'Toplam Sipariş',
      value: stats?.totalOrders || 0,
      change: '+8.2%',
      icon: ShoppingCart,
      color: 'blue'
    },
    {
      title: 'Aktif Masa',
      value: '12/20',
      change: '+2',
      icon: Users,
      color: 'green'
    },
    {
      title: 'Stok Uyarısı',
      value: '3',
      change: '-1',
      icon: Package,
      color: 'yellow'
    }
  ];

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Dashboard</h1>
          <p className="text-gray-600">Restoran performansınızı takip edin</p>
        </div>
        <div className="flex items-center space-x-2 text-sm text-gray-500">
          <Calendar size={16} />
          <span>{new Date().toLocaleDateString('tr-TR')}</span>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {statCards.map((stat, index) => {
          const Icon = stat.icon;
          return (
            <div key={index} className="stat-card">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-white/80 text-sm font-medium">{stat.title}</p>
                  <p className="text-2xl font-bold text-white mt-1">{stat.value}</p>
                  <div className="flex items-center mt-2">
                    <TrendingUp size={14} className="text-green-300 mr-1" />
                    <span className="text-green-300 text-sm">{stat.change}</span>
                  </div>
                </div>
                <div className="w-12 h-12 bg-white/20 rounded-lg flex items-center justify-center">
                  <Icon size={24} className="text-white" />
                </div>
              </div>
            </div>
          );
        })}
      </div>

      {/* Charts Section */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Sales Chart */}
        <div className="card">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900">Haftalık Satışlar</h3>
            <BarChart3 size={20} className="text-primary-600" />
          </div>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={salesData}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f3f4f6" />
              <XAxis dataKey="name" stroke="#6b7280" />
              <YAxis stroke="#6b7280" />
              <Tooltip
                contentStyle={{
                  backgroundColor: '#fff',
                  border: '1px solid #e5e7eb',
                  borderRadius: '8px',
                }}
              />
              <Line
                type="monotone"
                dataKey="sales"
                stroke="#ef4444"
                strokeWidth={3}
                dot={{ fill: '#ef4444', strokeWidth: 2, r: 4 }}
              />
            </LineChart>
          </ResponsiveContainer>
        </div>

        {/* Category Distribution */}
        <div className="card">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900">Kategori Dağılımı</h3>
            <Activity size={20} className="text-primary-600" />
          </div>
          <ResponsiveContainer width="100%" height={300}>
            <PieChart>
              <Pie
                data={categoryData}
                cx="50%"
                cy="50%"
                innerRadius={60}
                outerRadius={100}
                paddingAngle={5}
                dataKey="value"
              >
                {categoryData.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={entry.color} />
                ))}
              </Pie>
              <Tooltip
                contentStyle={{
                  backgroundColor: '#fff',
                  border: '1px solid #e5e7eb',
                  borderRadius: '8px',
                }}
              />
            </PieChart>
          </ResponsiveContainer>
          <div className="mt-4 grid grid-cols-2 gap-2">
            {categoryData.map((item, index) => (
              <div key={index} className="flex items-center space-x-2">
                <div
                  className="w-3 h-3 rounded-full"
                  style={{ backgroundColor: item.color }}
                />
                <span className="text-sm text-gray-600">{item.name}</span>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Recent Activity */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Recent Orders */}
        <div className="card">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Son Siparişler</h3>
          <div className="space-y-3">
            {recentOrders?.map((order: any) => (
              <div key={order.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <div>
                  <p className="font-medium text-gray-900">#{order.orderNumber}</p>
                  <p className="text-sm text-gray-600">{order.table?.number || 'Paket'}</p>
                </div>
                <div className="text-right">
                  <p className="font-medium text-primary-600">₺{order.totalAmount}</p>
                  <p className="text-sm text-gray-500">{order.status}</p>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Top Products */}
        <div className="card">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Popüler Ürünler</h3>
          <div className="space-y-3">
            {topProducts?.map((product: any) => (
              <div key={product.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <div className="flex items-center space-x-3">
                  <div className="w-10 h-10 bg-primary-100 rounded-lg flex items-center justify-center">
                    <Package size={16} className="text-primary-600" />
                  </div>
                  <div>
                    <p className="font-medium text-gray-900">{product.name}</p>
                    <p className="text-sm text-gray-600">{product.category?.name}</p>
                  </div>
                </div>
                <p className="font-medium text-primary-600">₺{product.price}</p>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
} 