import { PrismaClient, UserRole } from '@prisma/client';
import bcrypt from 'bcryptjs';

const prisma = new PrismaClient();

async function main() {
  // Demo admin kullanıcı
  const adminPassword = await bcrypt.hash('123456', 10);
  await prisma.user.upsert({
    where: { email: 'admin@espiropos.com' },
    update: {},
    create: {
      name: 'Admin',
      email: 'admin@espiropos.com',
      password: adminPassword,
      role: UserRole.ADMIN,
      isActive: true,
    },
  });

  // Kategoriler
  const categories = await prisma.category.createMany({
    data: [
      { name: 'Ana Yemekler', description: 'Ana yemek çeşitleri', sortOrder: 1 },
      { name: 'İçecekler', description: 'Soğuk ve sıcak içecekler', sortOrder: 2 },
      { name: 'Tatlılar', description: 'Tatlı ve atıştırmalıklar', sortOrder: 3 },
    ],
    skipDuplicates: true,
  });

  // Ürünler
  const anaYemek = await prisma.category.findFirst({ where: { name: 'Ana Yemekler' } });
  const icecek = await prisma.category.findFirst({ where: { name: 'İçecekler' } });
  const tatli = await prisma.category.findFirst({ where: { name: 'Tatlılar' } });

  await prisma.product.createMany({
    data: [
      { name: 'Izgara Tavuk', price: 180, cost: 90, categoryId: anaYemek?.id || '', isActive: true },
      { name: 'Köfte', price: 200, cost: 100, categoryId: anaYemek?.id || '', isActive: true },
      { name: 'Kola', price: 40, cost: 10, categoryId: icecek?.id || '', isActive: true },
      { name: 'Fanta', price: 40, cost: 10, categoryId: icecek?.id || '', isActive: true },
      { name: 'Baklava', price: 60, cost: 25, categoryId: tatli?.id || '', isActive: true },
    ],
    skipDuplicates: true,
  });

  // Masalar
  for (let i = 1; i <= 10; i++) {
    await prisma.table.upsert({
      where: { number: i },
      update: {},
      create: {
        number: i,
        capacity: i % 4 + 2,
        status: 'AVAILABLE',
        isActive: true,
      },
    });
  }

  // Stok
  const products = await prisma.product.findMany();
  for (const product of products) {
    await prisma.inventoryItem.upsert({
      where: { productId: product.id },
      update: {},
      create: {
        productId: product.id,
        quantity: 100,
        minQuantity: 10,
        maxQuantity: 200,
        cost: product.cost,
        unit: 'adet',
      },
    });
  }

  // Tedarikçi
  await prisma.supplier.upsert({
    where: { name: 'Demo Tedarikçi' },
    update: {},
    create: {
      name: 'Demo Tedarikçi',
      contactPerson: 'Ahmet Yılmaz',
      phone: '5551234567',
      email: 'tedarikci@demo.com',
      address: 'İstanbul',
      isActive: true,
    },
  });

  // Müşteri
  await prisma.customer.upsert({
    where: { phone: '5551112233' },
    update: {},
    create: {
      name: 'Demo Müşteri',
      phone: '5551112233',
      email: 'musteri@demo.com',
      address: 'Ankara',
      loyaltyPoints: 50,
      isActive: true,
    },
  });

  console.log('Demo kullanıcı ve örnek veriler başarıyla eklendi!');
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  }); 