import { Router } from 'express';
import { PrismaClient } from '@prisma/client';
import { z } from 'zod';

const router = Router();
const prisma = new PrismaClient();

const tableSchema = z.object({
  number: z.number().positive(),
  capacity: z.number().positive(),
  status: z.enum(['AVAILABLE', 'OCCUPIED', 'RESERVED', 'MAINTENANCE']).default('AVAILABLE'),
  isActive: z.boolean().default(true)
});

// Get all tables
router.get('/', async (req, res) => {
  try {
    const tables = await prisma.table.findMany({
      where: { isActive: true },
      orderBy: { number: 'asc' }
    });

    res.json({
      success: true,
      data: tables
    });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Create table
router.post('/', async (req, res) => {
  try {
    const data = tableSchema.parse(req.body);

    const table = await prisma.table.create({
      data
    });

    res.status(201).json({
      success: true,
      data: table
    });
  } catch (error) {
    if (error instanceof z.ZodError) {
      return res.status(400).json({ message: 'Validation error', errors: error.errors });
    }
    res.status(500).json({ message: 'Server error' });
  }
});

// Update table status
router.patch('/:id/status', async (req, res) => {
  try {
    const { id } = req.params;
    const { status } = req.body;

    const table = await prisma.table.update({
      where: { id },
      data: { status }
    });

    res.json({
      success: true,
      data: table
    });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

export default router; 