import { Router } from 'express';
import { PrismaClient } from '@prisma/client';
import { z } from 'zod';

const router = Router();
const prisma = new PrismaClient();

const reservationSchema = z.object({
  tableId: z.string(),
  customerName: z.string().min(1),
  customerPhone: z.string().optional(),
  customerEmail: z.string().email().optional(),
  date: z.string(),
  time: z.string(),
  partySize: z.number().positive(),
  notes: z.string().optional()
});

// Get all reservations
router.get('/', async (req, res) => {
  try {
    const { date, status } = req.query;
    
    const where: any = {};
    
    if (date) {
      const targetDate = new Date(date as string);
      const startDate = new Date(targetDate);
      startDate.setHours(0, 0, 0, 0);
      const endDate = new Date(targetDate);
      endDate.setHours(23, 59, 59, 999);
      
      where.date = {
        gte: startDate,
        lte: endDate
      };
    }
    
    if (status) {
      where.status = status;
    }

    const reservations = await prisma.reservation.findMany({
      where,
      include: {
        table: true,
        user: {
          select: { id: true, name: true }
        }
      },
      orderBy: { date: 'asc' }
    });

    res.json({
      success: true,
      data: reservations
    });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Create reservation
router.post('/', async (req, res) => {
  try {
    const data = reservationSchema.parse(req.body);

    const reservation = await prisma.reservation.create({
      data: {
        ...data,
        date: new Date(data.date)
      },
      include: {
        table: true,
        user: {
          select: { id: true, name: true }
        }
      }
    });

    res.status(201).json({
      success: true,
      data: reservation
    });
  } catch (error) {
    if (error instanceof z.ZodError) {
      return res.status(400).json({ message: 'Validation error', errors: error.errors });
    }
    res.status(500).json({ message: 'Server error' });
  }
});

export default router; 