import { Router } from 'express';
import { PrismaClient } from '@prisma/client';
import QRCode from 'qrcode';

const router = Router();
const prisma = new PrismaClient();

// Get menu for QR code (public)
router.get('/menu', async (req, res) => {
  try {
    const categories = await prisma.category.findMany({
      where: { isActive: true },
      include: {
        products: {
          where: { isActive: true },
          orderBy: { name: 'asc' }
        }
      },
      orderBy: { sortOrder: 'asc' }
    });

    res.json({
      success: true,
      data: {
        categories,
        restaurant: {
          name: 'Espiropos Restaurant',
          description: 'Modern restoran yönetim sistemi',
          logo: '/logo.png'
        }
      }
    });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Generate QR code for menu
router.get('/generate', async (req, res) => {
  try {
    const menuUrl = `${process.env.QR_BASE_URL || 'http://localhost:3000'}/menu`;
    
    const qrCodeDataUrl = await QRCode.toDataURL(menuUrl, {
      width: 300,
      margin: 2,
      color: {
        dark: '#000000',
        light: '#FFFFFF'
      }
    });

    res.json({
      success: true,
      data: {
        qrCode: qrCodeDataUrl,
        menuUrl
      }
    });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Get menu by table number
router.get('/table/:tableNumber', async (req, res) => {
  try {
    const { tableNumber } = req.params;

    const table = await prisma.table.findFirst({
      where: { 
        number: parseInt(tableNumber),
        isActive: true
      }
    });

    if (!table) {
      return res.status(404).json({ message: 'Table not found' });
    }

    const categories = await prisma.category.findMany({
      where: { isActive: true },
      include: {
        products: {
          where: { isActive: true },
          orderBy: { name: 'asc' }
        }
      },
      orderBy: { sortOrder: 'asc' }
    });

    res.json({
      success: true,
      data: {
        table,
        categories,
        restaurant: {
          name: 'Espiropos Restaurant',
          description: 'Modern restoran yönetim sistemi',
          logo: '/logo.png'
        }
      }
    });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

export default router; 