import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import morgan from 'morgan';
import dotenv from 'dotenv';
import path from 'path';

// Routes
import authRoutes from './routes/auth';
import productRoutes from './routes/products';
import categoryRoutes from './routes/categories';
import orderRoutes from './routes/orders';
import tableRoutes from './routes/tables';
import reservationRoutes from './routes/reservations';
import inventoryRoutes from './routes/inventory';
import supplierRoutes from './routes/suppliers';
import transactionRoutes from './routes/transactions';
import customerRoutes from './routes/customers';
import qrRoutes from './routes/qr';
import reportRoutes from './routes/reports';

// Middleware
import { errorHandler } from './middleware/errorHandler';
import { authMiddleware } from './middleware/auth';

dotenv.config();

const app = express();
const PORT = process.env.PORT || 5000;

// Middleware
app.use(helmet());
app.use(cors());
app.use(morgan('combined'));
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true }));

// Static files
app.use('/uploads', express.static(path.join(__dirname, '../uploads')));

// Health check
app.get('/health', (req, res) => {
  res.json({ status: 'OK', timestamp: new Date().toISOString() });
});

// Public routes
app.use('/api/auth', authRoutes);
app.use('/api/qr', qrRoutes);

// Protected routes
app.use('/api/products', authMiddleware, productRoutes);
app.use('/api/categories', authMiddleware, categoryRoutes);
app.use('/api/orders', authMiddleware, orderRoutes);
app.use('/api/tables', authMiddleware, tableRoutes);
app.use('/api/reservations', authMiddleware, reservationRoutes);
app.use('/api/inventory', authMiddleware, inventoryRoutes);
app.use('/api/suppliers', authMiddleware, supplierRoutes);
app.use('/api/transactions', authMiddleware, transactionRoutes);
app.use('/api/customers', authMiddleware, customerRoutes);
app.use('/api/reports', authMiddleware, reportRoutes);

// Error handling
app.use(errorHandler);

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({ message: 'Route not found' });
});

app.listen(PORT, () => {
  console.log(`🚀 Server running on port ${PORT}`);
  console.log(`📊 Environment: ${process.env.NODE_ENV}`);
  console.log(`🔗 Health check: http://localhost:${PORT}/health`);
}); 